import EventEmitter from "eventemitter3";
import { RedisClientType, createClient } from "redis";
import {
  InstanceBanEvent,
  InstanceUnbanEvent,
  ShadowEvent,
  UserBanEvent,
  UserUnbanEvent,
} from "./events";

interface Events {
  user_ban: (info: UserBanEvent) => void;
  user_unban: (info: UserUnbanEvent) => void;
  instance_ban: (info: InstanceBanEvent) => void;
  instance_unban: (info: InstanceUnbanEvent) => void;
}

export class ShadowAPI extends EventEmitter<Events> {
  private ready: boolean = false;
  private shadow_host: string;
  private redis_sub: RedisClientType;

  private RedisEvents = [
    UserBanEvent,
    UserUnbanEvent,
    InstanceBanEvent,
    InstanceUnbanEvent,
  ] as const;

  private constructor(shadow_host: string, redis_uri: string) {
    super();

    this.shadow_host = shadow_host;

    this.redis_sub = createClient({ url: redis_uri });
  }

  /**
   * Check shadow_host to make sure it's a shadow endpoint
   * @returns
   */
  async probeHost(): Promise<boolean> {
    // TODO: check shadow host and verify it's running & is a shadow endpoint
    return true;
  }

  async connectToRedis() {
    await this.redis_sub.connect();
    this.setupRedisSubscriptions();
  }

  async connect() {
    if (this.ready) return;

    await this.probeHost();
    await this.connectToRedis();

    this.ready = true;
  }

  setupRedisSubscriptions() {
    // TODO: maybe dynamically create this?
    // or the Events interface?
    const classMapping: [(typeof this.RedisEvents)[number], keyof Events][] = [
      [UserBanEvent, "user_ban"],
      [UserUnbanEvent, "user_unban"],
      [InstanceBanEvent, "instance_ban"],
      [InstanceUnbanEvent, "instance_unban"],
    ];

    const prefix = "shadow:";

    for (const [eventcl, emit] of classMapping) {
      this.redis_sub.subscribe(prefix + eventcl.getChannel(), (data) => {
        this.emit(emit, eventcl.fromString(data) as any);
      });
    }
  }

  /**
   * Create shadow instance
   * @param shadow_host
   * @param redis_uri
   * @returns
   */
  static create(shadow_host: string, redis_uri: string): ShadowAPI {
    const shadow = new this(shadow_host, redis_uri);
    return shadow;
  }

  static getAllEvents(): ShadowEvent[] {
    return [UserBanEvent, UserUnbanEvent, InstanceBanEvent, InstanceUnbanEvent];
  }
}
