import { PluginOption, defineConfig, loadEnv } from "vite";
import react from "@vitejs/plugin-react-swc";

/**
 * Run middlewares during development
 * @param API_BACKEND_HOST
 * @returns
 */
function dev_middleware(API_BACKEND_HOST: string | undefined): PluginOption {
  return {
    name: "dev-middleware",
    config() {
      return {
        server: {
          allowedHosts: true,
        },
        preview: {},
      };
    },
    configureServer(server) {
      if (typeof API_BACKEND_HOST === "undefined") {
        throw new Error("DEV_BACKEND_HOST is not specified");
      }

      let data: any;

      server.middlewares.use("/_dev/ssdata", async (req, res) => {
        res.setHeader("Content-Type", "application/json");
        try {
          res.write(JSON.stringify(data));
        } catch (e) {
          res.write("{}");
        }
        res.end();
      });

      server.middlewares.use("/handoff", async (req, res, next) => {
        const sessionId = (req.url || "").slice(1).split("/")[0];

        const middle = await fetch(
          new URL("/handoff/" + sessionId, API_BACKEND_HOST),
          {
            redirect: "manual",
            headers: {
              "x-vite-middleware": "yes",
            },
          }
        );

        if (middle.headers.get("location")) {
          res.statusCode = 302;
          res.setHeader("Location", middle.headers.get("location")!.toString());
          res.end();
          return;
        }

        data = (await middle.json())?.data;

        next();
      });

      // the backend has interaction middleware to sync login sessions w/ oidc-provider
      // this adds the middleware to vite
      server.middlewares.use("/interaction", async (req, res, next) => {
        // grab the interaction ID from the url (which has the format of /interaction/:uid)
        const interactionId = (req.url || "").slice(1).split("/")[0];

        // send request to backend using development endpoint
        // (only exposed when backend is NODE_ENV===development)
        const middle = await fetch(
          API_BACKEND_HOST + "/_dev/interaction/" + interactionId,
          {
            method: "POST",
            redirect: "manual",
            headers: {
              // we need to pass the cookies the client is using to keep sessions
              cookie: req.headers.cookie || "",
            },
          }
        );

        // if the middleware is redirecting we need to follow the redirect
        if (middle.headers.get("Location")) {
          const location = new URL(
            middle.headers.get("Location")!,
            "http://" + req.headers.host
          );

          if (location.host !== req.headers.host) {
            // sometimes the backend will send a redirect including the backend's port,
            // which will cause an error in development mode due to the backend not serving the client
            location.host = req.headers.host!;
          }

          // use a temporary redirect, like what the backend would send
          res.statusCode = 302;
          res.setHeader("Location", location.toString());
          res.end();
          return;
        }

        // something errored in the backend
        if (middle.status !== 200) {
          console.log(await middle.text());
          res.write(
            "Backend internal error, check console (development message)"
          );
          res.end();
          return;
        }

        // the middleware didn't need to do anything, resume normal rendering
        next();
      });
    },
  };
}

// https://vitejs.dev/config/
export default defineConfig(({ mode }) => {
  const env = loadEnv(mode, process.cwd(), "");

  if (env.DEV_BACKEND_HOST) {
    return {
      plugins: [react(), dev_middleware(env.DEV_BACKEND_HOST)],
      server: {
        proxy: {
          "/api": env.DEV_BACKEND_HOST,

          // CSRF handling
          "/logout": env.DEV_BACKEND_HOST,

          // fediverse endpoints
          "/.well-known": env.DEV_BACKEND_HOST,
          "/x": env.DEV_BACKEND_HOST,
          "/inbox": env.DEV_BACKEND_HOST,
        },
      },
    };
  }

  return {
    plugins: [react()],
  };
});
