import { LoadingButton } from "@mui/lab";
import {
  Alert,
  Box,
  Button,
  Card,
  CardContent,
  CircularProgress,
  Dialog,
  DialogActions,
  DialogContent,
  DialogTitle,
  Stack,
  Step,
  StepContent,
  StepLabel,
  Stepper,
  TextField,
  Typography,
} from "@mui/material";
import { useEffect, useRef, useState } from "react";
import { useNavigate, useSearchParams } from "react-router-dom";
import { api } from "../lib/utils";

enum LoginState {
  INSTANCE,
  USERNAME,
  CODE_INPUT,
}

export const LoginPage = () => {
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [loading, setLoading] = useState(false);
  const [showHow, setShowHow] = useState(false);
  const [state, setState] = useState<LoginState>(LoginState.INSTANCE);
  const [errors, setErrors] = useState<
    {
      type: "info" | "warning" | "error";
      text: string;
    }[]
  >([]);

  const instanceInput = useRef<HTMLInputElement>(null);
  const usernameInput = useRef<HTMLInputElement>(null);
  const codeInput = useRef<HTMLInputElement>(null);

  /**
   * Instance hostname
   */
  const [instance, setInstance] = useState("");
  /**
   * Localpart of the webfinger username
   */
  const [username, setUsername] = useState("");

  /**
   * Metadata used in UI elements
   */
  const [meta, setMeta] = useState<any>();
  /**
   * Used by CODE_SENT
   *
   * The code that was sent to the user, controls a textfield
   */
  const [code, setCode] = useState("");

  const handleError = (err: string | undefined) => {
    setErrors([
      {
        type: "error",
        text: err || "Unknown Error",
      },
    ]);
  };

  useEffect(() => {
    switch (state) {
      case LoginState.INSTANCE:
        instanceInput.current?.focus();
        break;
      case LoginState.USERNAME:
        usernameInput.current?.focus();
        break;
      case LoginState.CODE_INPUT:
        codeInput.current?.focus();
        break;
    }
  }, [state]);

  const doLogin = async () => {
    setErrors([]);

    const tryUsernameLogin = (username: string, instance: string) => {
      return api<
        | { success: false; error: string }
        | { success: true; step: string; data: any }
      >("/api/v1/login/step/username", "POST", {
        username,
        instance,
      }).then(({ status, data }) => {
        if (data.success) {
          if (data.data) setMeta(data.data);

          setState(LoginState.CODE_INPUT);
        } else {
          handleError(`[${status}] ` + (data.error || "unknown"));
        }

        return data;
      });
    };

    switch (state) {
      case LoginState.INSTANCE: {
        setLoading(true);

        let _username: string | undefined;
        let _instance = instance + "";

        while (_instance[0] === "@") {
          _instance = _instance.substring(1);
        }

        if (_instance.indexOf("@") > -1) {
          _username = _instance.split("@")[0];
          _instance = _instance.split("@")[1];
        }

        // must be done twice in the case someone types grant@@grants.cafe
        _instance = _instance.replace(/@/g, "");

        if (_username) setUsername(_username);
        setInstance(_instance);

        api<
          { success: false; error: string } | { success: true; step: string }
        >("/api/v1/login/step/instance", "POST", {
          domain: _instance,
        })
          .then(async ({ status, data }) => {
            if (data.success) {
              // TODO: add OIDC support

              if (_username) {
                await tryUsernameLogin(_username, _instance);
              } else {
                setState(LoginState.USERNAME);
              }
            } else {
              handleError(`[${status}] ` + (data.error || "unknown"));
            }
          })
          .finally(() => {
            setLoading(false);
          });
        break;
      }
      case LoginState.USERNAME: {
        setLoading(true);

        tryUsernameLogin(username, instance).finally(() => {
          setLoading(false);
        });
        break;
      }
      case LoginState.CODE_INPUT: {
        setLoading(true);

        api("/api/v1/login/step/verify", "POST", {
          code,
        })
          .then(({ status, data }) => {
            if (data.success) {
              // NOTE: possible issue; redirecting unprotected
              if (searchParams.has("return")) {
                // do not use navigate() to bypass history navigation
                // we need the backend to preprocess the request
                window.location.href = searchParams.get("return")!;
              } else {
                navigate("/");
              }
            } else {
              handleError(`[${status}] ` + (data.error || "unknown"));
            }
          })
          .finally(() => setLoading(false));
        break;
      }
    }
  };

  return (
    <Box
      sx={{
        display: "flex",
        justifyContent: "center",
        alignItems: "center",
        p: 5,
      }}
    >
      <Box sx={{ width: "min(400px,90%)" }}>
        <Card>
          <CardContent>
            <Typography variant="h5" component="div">
              Fediverse Login
            </Typography>

            <Typography component="div">
              Welcome! Enter your Fediverse identity below to continue verifying
              who you are.
            </Typography>

            <Button variant="text" onClick={() => setShowHow(true)}>
              How does this work?
            </Button>

            <Stack direction="column" gap={1} sx={{ mt: 1 }}>
              {errors.map((err) => (
                <Alert key={JSON.stringify(err)} severity={err.type}>
                  {err.text}
                </Alert>
              ))}

              {state === LoginState.INSTANCE && (
                <>
                  <TextField
                    label="Instance or full handle"
                    placeholder="grants.cafe"
                    value={instance}
                    autoFocus
                    inputRef={instanceInput}
                    onChange={(e) => setInstance(e.target.value)}
                    onKeyUp={(e) => e.key === "Enter" && doLogin()}
                  />
                  <Typography variant="caption" color="text.secondary">
                    Pro tip: use your full handle (grant@toast.ooo)
                  </Typography>
                </>
              )}
              {state === LoginState.USERNAME && (
                <>
                  <TextField
                    label="Username"
                    placeholder="grant"
                    value={username}
                    autoFocus
                    inputRef={usernameInput}
                    onChange={(e) => setUsername(e.target.value)}
                    onKeyUp={(e) => e.key === "Enter" && doLogin()}
                    InputProps={{
                      startAdornment: (
                        <Typography color="text.secondary">{"@"}</Typography>
                      ),
                      endAdornment: (
                        <Typography color="text.secondary">
                          {"@" + instance}
                        </Typography>
                      ),
                    }}
                  />
                </>
              )}
              {state === LoginState.CODE_INPUT && (
                <>
                  <Alert severity="info">
                    You should receive a message from <b>{meta.account}</b> with
                    your code
                  </Alert>
                  <TextField
                    label="The code you got sent"
                    placeholder="12345"
                    value={code}
                    autoFocus
                    inputRef={codeInput}
                    type="number"
                    inputProps={{
                      pattern: "[0-9]*",
                    }}
                    onChange={(e) => setCode(e.target.value)}
                    onKeyUp={(e) => e.key === "Enter" && doLogin()}
                  />
                </>
              )}
            </Stack>

            <Box sx={{ display: "flex", mt: 1 }}>
              {state !== LoginState.INSTANCE && (
                <Button
                  variant="outlined"
                  onClick={() => setState(LoginState.INSTANCE)}
                >
                  Start Over
                </Button>
              )}
              <Box sx={{ flexGrow: 1 }} />
              <LoadingButton
                loading={loading}
                variant="contained"
                onClick={doLogin}
              >
                Next
              </LoadingButton>
            </Box>
          </CardContent>
        </Card>
      </Box>

      <Dialog open={loading}>
        <DialogContent>
          <CircularProgress />
        </DialogContent>
      </Dialog>

      <Dialog open={showHow} onClose={() => setShowHow(false)}>
        <DialogTitle>How does this work?</DialogTitle>
        <DialogContent>
          <Stepper activeStep={-1} orientation="vertical">
            <Step expanded>
              <StepLabel>Identify Your Instance</StepLabel>
              <StepContent>
                A simple text box, we will verify it for you
              </StepContent>
            </Step>
            <Step expanded>
              <StepLabel>Identify Your User</StepLabel>
              <StepContent>
                Inputing the username you go by on your instance
              </StepContent>
            </Step>
            <Step expanded>
              <StepLabel>Send/Receive a Code</StepLabel>
              <StepContent>
                Sending or receiving depends on the instance's software
              </StepContent>
            </Step>
          </Stepper>
          <Alert color="info" sx={{ mt: 1 }}>
            No passwords will be entered
          </Alert>
        </DialogContent>
        <DialogActions>
          <Button
            variant="contained"
            color="success"
            sx={{ textTransform: "unset" }}
            onClick={() => setShowHow(false)}
          >
            Ok, sounds good!
          </Button>
        </DialogActions>
      </Dialog>
    </Box>
  );
};
