import { useEffect, useState } from "react";
import { api } from "../lib/utils";
import { useParams } from "react-router-dom";
import {
  Alert,
  Box,
  Card,
  CardContent,
  CircularProgress,
  Dialog,
  DialogContent,
  Stack,
  Typography,
} from "@mui/material";
import { IOIDC_Client, IOIDC_Interaction } from "../types/oidc";
import { ClientDetailsCard } from "./ClientDetailsCard";
import { UserInfoCard } from "./UserInfoCard";
import { LoadingButton } from "@mui/lab";

export const InteractionPage = () => {
  const { id: interactionId } = useParams();
  const [loading, setLoading] = useState(true);
  const [loading_approve, setLoading_Approve] = useState(false);
  const [loading_deny, setLoading_Deny] = useState(false);
  const [fatalError, setFatalError] = useState<string>();
  const [error, setError] = useState<{
    type: "shadow" | "error";
    message: string;
  }>();

  const [interaction, setInteraction] = useState<IOIDC_Interaction>();
  const [user, setUser] = useState<{
    sub: string;
    handle: `${string}@${string}`;
  }>();
  const [client, setClient] = useState<IOIDC_Client>();

  const refreshInteraction = (interactionId: string) => {
    return api<IOIDC_Interaction>(
      "/api/v1/interaction?id=" + interactionId
    ).then((interactionReq) => {
      if (interactionReq.status === 200) {
        setInteraction(interactionReq.data);
        return interactionReq.data;
      } else {
        setFatalError((interactionReq.data as any).error || "Unknown Error");
      }
    });
  };

  const refreshClient = (client_id: string) => {
    return api<IOIDC_Client>("/api/v1/client?id=" + client_id).then(
      (clientReq) => {
        if (clientReq.status === 200) {
          setClient(clientReq.data);
          return clientReq.data;
        } else {
          setFatalError((clientReq.data as any).error || "Unknown Error");
        }
      }
    );
  };

  useEffect(() => {
    if (!interactionId) {
      throw new Error(
        "Interaction ID is not set? This should not be possible."
      );
    }

    Promise.allSettled([
      api("/api/v1/whoami").then((whoami) => {
        setUser("error" in whoami.data ? undefined : whoami.data);
      }),
      refreshInteraction(interactionId).then((interaction) => {
        if (interaction) {
          return refreshClient(interaction.params.client_id);
        }
      }),
    ]).finally(() => {
      setLoading(false);
    });
  }, []);

  const stopAllLoading = () => {
    setLoading(false);
    setLoading_Approve(false);
    setLoading_Deny(false);
  };

  const doApprove = async () => {
    setLoading_Approve(true);
    api<
      | { success: true; returnTo: string }
      | { success: false; error: string; metadata?: any }
    >("/api/v1/interaction/" + interactionId + "/confirm", "POST")
      .then(({ status, data }): any => {
        if (status === 200 && data.success) {
          window.open(data.returnTo, "_self");
        } else {
          if (data.success) {
            setError({
              type: "error",
              message: "Unknown error",
            });
          } else {
            if (data.error === "shadow") {
              setError({
                type: "shadow",
                message: data.metadata?.message || "Unknown Error",
              });
            } else {
              setError({
                type: "error",
                message: data.error,
              });
            }
          }
        }
      })
      .finally(() => {
        stopAllLoading();
      });
  };

  const doDeny = async () => {
    setLoading_Deny(true);
    api<
      | { success: true; returnTo: string }
      | { success: false; error: string; metadata?: any }
    >("/api/v1/interaction/" + interactionId + "/abort", "POST")
      .then((data) => {
        if (data.status === 200 && data.data.success) {
          window.open(data.data.returnTo, "_self");
        } else {
          setError((data.data as any).error);
        }
      })
      .finally(() => {
        stopAllLoading();
      });
  };

  return (
    <Box
      sx={{
        display: "flex",
        justifyContent: "center",
        alignItems: "center",
        p: 5,
      }}
    >
      <Box sx={{ width: "min(400px,90%)" }}>
        <Card>
          <CardContent>
            <Typography variant="h5" component="div">
              Fediverse Auth
            </Typography>

            {error?.type === "error" && (
              <Alert severity="error">{error.message}</Alert>
            )}

            {error?.type === "shadow" && (
              <Alert severity="warning">
                <strong>Shadow Error</strong>
                <br />
                {error.message}
              </Alert>
            )}

            {client && <ClientDetailsCard client={client} sx={{ mt: 1 }} />}
            {user ? (
              <UserInfoCard user={user} sx={{ mt: 1 }} />
            ) : (
              <Alert severity="warning">User session lost</Alert>
            )}

            <Typography sx={{ mt: 1 }}>
              <b>{client?.clientName || "No Name"}</b> is wanting to identify
              you and redirect you to{" "}
              <code>{interaction?.params.redirect_uri}</code>
            </Typography>

            <Stack direction="column" gap={1} sx={{ mt: 1 }}>
              <LoadingButton
                loading={loading_approve}
                disabled={loading_deny || loading}
                variant="contained"
                color="success"
                onClick={doApprove}
              >
                Approve
              </LoadingButton>
              <LoadingButton
                loading={loading_deny}
                disabled={loading_approve || loading}
                variant="outlined"
                color="error"
                onClick={doDeny}
              >
                Deny
              </LoadingButton>
            </Stack>
          </CardContent>
        </Card>
      </Box>

      <Dialog open={loading}>
        <DialogContent>
          <CircularProgress />
        </DialogContent>
      </Dialog>

      <Dialog open={Boolean(fatalError)}>
        <DialogContent>
          <Typography variant="h5">Fatal Error</Typography>
          {fatalError}
        </DialogContent>
      </Dialog>
    </Box>
  );
};
