import { Alert, Link, Typography } from "@mui/material";
import { useSSData } from "../lib/ServerSideData";
import { PageWrapper } from "../PageWrapper";
import { DocLinks } from "../lib/DocLinks";
import { Header } from "../components/Header";

type Data =
  | { state: "NOT_FOUND" }
  | { state: "MISSING_RETURN" | "INVALID_RETURN" }
  | { state: "INTERNAL_PROFILE_ERROR" };

export const HandoffPage = () => {
  const ssdata = useSSData<Data>();

  switch (ssdata.state) {
    case "NOT_FOUND":
      return <InfoPage />;
    case "MISSING_RETURN":
    case "INVALID_RETURN":
      return <ReturnError error={ssdata.state} />;
    case "INTERNAL_PROFILE_ERROR":
      return <InternalError />;
  }

  return <div>UNKNOWN STATE {JSON.stringify(ssdata)}</div>;
};

const InfoPage = () => {
  return (
    <PageWrapper>
      <Header />

      <Alert severity="info" sx={{ mb: 1 }}>
        <strong>The handoff link you followed does not exist</strong>
        <br />
        ...or the session expired
      </Alert>

      <Typography sx={{ mb: 1 }}>
        This link was sent to you as a response to a third party app requesting
        it on your behalf.
        <br />
        <br />
        The links created are single use and used to prove your identity,
        allowing for easy signin on cooperating apps
      </Typography>

      <ul>
        <li>
          <Link href="/">What is Fediverse Auth?</Link>
        </li>
        <li>
          <Link href={DocLinks.HANDOFF_DOCS}>Handoff Documentation</Link>
        </li>
      </ul>
    </PageWrapper>
  );
};

const ReturnError = ({
  error,
}: {
  error: "MISSING_RETURN" | "INVALID_RETURN";
}) => {
  return (
    <PageWrapper>
      <Header />

      {error === "MISSING_RETURN" ? (
        <Alert severity="error">
          <strong>Missing return URL</strong>
          <br />
          ?return_uri needs to be set on this request
        </Alert>
      ) : (
        <Alert severity="error">
          <strong>Return URL is invalid</strong>
          <br />
          ?return_uri needs to be an absolute URL
        </Alert>
      )}
    </PageWrapper>
  );
};

const InternalError = () => {
  return (
    <PageWrapper>
      <Header />

      <Alert severity="error">Internal error, cannot continue</Alert>
    </PageWrapper>
  );
};
