import { NodeInfo } from "../types/nodeinfo.js";
import { IOIDC_Public_Client } from "../types/oidc.js";
import { getNodeInfo } from "./nodeinfo.js";
import { oidc } from "./oidc.js";
import { type Request } from "express";

/**
 * Domain name regex
 * @note this allows for domain names that don't have periods in them, unlikely, but possible
 * @see https://stackoverflow.com/a/26987741
 */
export const DOMAIN_REGEX =
  /^(((?!-))(xn--|_)?[a-z0-9-]{0,61}[a-z0-9]{1,1}\.)*(xn--)?([a-z0-9][a-z0-9\-]{0,60}|[a-z0-9-]{1,30}\.[a-z]{2,})$/g;

type IPrivateClient = Awaited<ReturnType<typeof oidc.Client.find>>;

export const makeClientPublic = (
  client: IPrivateClient
): IOIDC_Public_Client | undefined => {
  if (!client) return undefined;

  return {
    clientId: client.clientId,
    redirectUris: client.redirectUris!,
    clientName: client.clientName,
    clientUri: client.clientUri,
    logoUri: client.logoUri,
    policyUri: client.policyUri,
    tosUri: client.tosUri,
  };
};

/**
 * Verify if the instance at the domain is allowed
 *
 * Verifies:
 * 1. has a NodeInfo
 * 2. the NodeInfo is a supported version (2.0|2.1)
 * 3. the NodeInfo announces activitypub support
 *
 * @param instance_hostname
 */
export const isInstanceDomainValid = async (
  instance_hostname: string
): Promise<boolean> => {
  let nodeinfo: NodeInfo;
  try {
    nodeinfo = await getNodeInfo(instance_hostname);
  } catch (e) {
    return false;
  }

  if (nodeinfo.version !== "2.0" && nodeinfo.version !== "2.1") {
    return false;
  }

  return nodeinfo.protocols.indexOf("activitypub") > -1;
};

/**
 * Get a safe URL
 *
 * This restricts the protocol and that's basically it
 *
 * This could be improved to proxy all requests
 *
 * @param unsafe_url
 * @returns
 */
export const getSafeURL = (unsafe_url: string): string | undefined => {
  if (unsafe_url.indexOf("https://") !== 0) return undefined;

  return unsafe_url;
};

export const getExpressIP = (req: Request): string => {
  if (process.env.NODE_ENV === "production") {
    let ip: string | undefined;

    if (typeof req.headers["x-forwarded-for"] === "string") {
      ip = req.headers["x-forwarded-for"];
    } else {
      ip = req.headers["x-forwarded-for"]?.[0];
    }

    return ip || req.socket.remoteAddress!;
  }

  return req.socket.remoteAddress!;
};
