/**
 * Shadow moderation API utilities
 */

// TODO: Redis subscriptions for ban notifications

interface IShadowAPI {
  /**
   * Check if a user can login
   *
   * This should be expected to be called frequently
   * Cache this and bust it with Redis subscription
   *
   * @throws UserLoginError on failure
   * @param user
   * @returns true
   */
  canUserLogin(user: IShadowUser): Promise<true>;
}

class ShadowAPI_ implements IShadowAPI {
  private async api<T>(
    endpoint: `/${string}`,
    method = "GET",
    body?: any
  ): Promise<{ status: number; data: T }> {
    let headers: { [k: string]: string } = {
      Authorization: "Bearer " + process.env.SHADOW_TOKEN,
    };
    let params: RequestInit = {
      method,
    };

    if (typeof body !== "undefined") {
      headers["Content-Type"] = "application/json";
      params.body = JSON.stringify(body);
    }

    params.headers = headers;

    const req = await fetch(
      process.env.SHADOW_HOST! + "/api/fediverse-auth/v1" + endpoint,
      params
    );

    const res = await req.json();

    return {
      status: req.status,
      data: res as any,
    };
  }

  async canUserLogin(user: IShadowUser): Promise<true> {
    const { status, data } = await this.api<{
      can_login: boolean;
      reason?: string;
    }>("/login", "POST", {
      sub: user.sub,
      ip: user.ip,
    });

    if (status === 200 && data.can_login) {
      return true;
    } else {
      throw new UserLoginError(data?.reason || "Unknown error");
    }
  }
}

export class UserLoginError extends Error {
  constructor(reason: string) {
    super(reason);
    this.name = "UserLoginError";
  }
}

/**
 * The SHADOW_HOST environment variable isn't set
 *
 * Assume every login is permitted
 */
class ShadowAPI_Masked implements IShadowAPI {
  async canUserLogin(user: IShadowUser): Promise<true> {
    return true;
  }
}

interface IShadowUser {
  sub: string;
  ip: string;
}

export const ShadowAPI = process.env.SHADOW_HOST
  ? new ShadowAPI_()
  : new ShadowAPI_Masked();
