import Provider from "oidc-provider";
import fs from "node:fs";
import { PrismaAdapter } from "./adapter.js";
import { Issuer } from "openid-client";
import { IInstance, getInstanceMeta } from "./instance/instanceMeta.js";
import { IProfile, getUserMeta } from "./instance/userMeta.js";
import { ShadowAPI, UserLoginError } from "./shadow.js";
import { APub } from "./apub/utils.js";

/**
 * ⚠ DEVELOPMENT KEYS ⚠
 *
 * These should not be used in production
 */
const DEV_KEYS = {
  // oidc-provider jwk is required
  jwks: [
    {
      p: "5uVHvbpHt2dlAn5_JIDXWmXIw_GkOJubXbHq8MY-398oY36aKOqr0ej4qIYO2dfzuJbXqLPDwRjp0t-KJZ8I7AZUbDAolF9x4bAOY1k5_t1BcI1GJZb6heK3Rqk3us_LiMSIhbTJYGZq8HTlabqcVAau2KhcCpYQUOJvHCVnzZs",
      kty: "RSA",
      q: "u9EmNAX0fxNPsUjMT-V4nSH_dSi5-iXNkAg-5ANOcivaggRpRdxPIN8d074guGFdsgWrejFBkj-8zS0UcG_3E78TOj7ZAXFjd08CVCTyKaKlkDu94_p746pc4iVOieIWFlNwajqbfuJfaFXftwd6G43-cDav-fcaMg4vtJGyybE",
      d: "cMA1KBZAq0hfuRU6q8j1nuXLRMmgWeKtA97z8oZyq2YW_WqHFX7dMWi5-o1qLXU0BcLCQOM5PEdH9KXtstk-8Oqf5NGpLJ0Fy_2l6ANXPyoMhk0wKag2EFQwU5Dyl-4RgDWLqXelF_u6arFSoWUmAuT-ZJkMfPmJp0Eo13TkzYviFX-sGsuNjhVgz1wL5YN0jvdup7XdEgnlFRbd6bf7TQ9AJsOXlJt4H4EARRh_XzFNn_y09_JIpvHFu7D6zPbAHZAj05smbuO6fMIYNOpT5uNgn0GINBwO3S-ZwmhClhpcDlLk8boLGrJI85Y6AqHF6r19Rl44EQADwLw2UceHYQ",
      e: "AQAB",
      use: "sig",
      kid: "sig-1715032435",
      qi: "A7QcvALvsMumHGyE59510xUBl4DZKCLjaXV8C6D-FxpXsQF8LlDYPbpJDbNo3_NbzCZzQ8VU3DbyKnb_RSlMw8RgV24_VyaojwkArij8twMtA5usUK5BFachCscAudgI_qR76HK_1KOLhyX5BMfpxiaJLzxi9aY6Vp8cR1-u7Tc",
      dp: "f47_eVOmNy6J4TzdJN-BGdHNfmLK5PMifDrEnswHBEsW1xCkPiKXVdotNX0KS1NAtCOxdOQLK2yGEReqDGq11R8SGMrqQD4SfipzaHNs1N6LPpDtxeqI8Np1gjYOMciGm0JoYeWksvsh7UHHVAfiQZGHmu44GykYTncqBxSrKi0",
      alg: "RS256",
      dq: "Sild4nDviDYB48kRFANSSwmfq413UjUXJGZ9Ht_HXAHA-FHxl6pUfHBdgLy0gtm_e4oNmeRVNgCA9qt0RKmRcHSkjP1ABvfVBMln6_3iuVoF8hwE8T55KP6eSpXcm0lw20P7QZb-y21rqvEts0H6j2LUM08E0bkm2NkNMUnOxSE",
      n: "qWYfDB2PM0H88fAwLS_kKGufLlnDWYKUJ0v-P4CDwI09PMGTxNrHVsfNpFyu7bcbdyhP-h3QWs3XYE-kB5-HTfmtoosuzgRkaIHT13dTGS24smtmiZ-9SZPp3Jh1eUT-z24-TvQwhT-gEnkmcER_Ee9UIDpRb0b2aMX89q9M8pOuxm0b0Jth2l2mlYPrVTOChabW6H-ekcXuX2c02CQeIOGmwes56bMyFmOia2e7WaQfmVPVhOp5SHqpHoR2IeIIqIC8QgixRpop5xJLlaOJ9QI5qa76iUn33oob30GgqngDuRMgBvT42lZX6TfS-R-ToqoamHFwGmXIw79yHfnbKw",
    },
  ],
  // oidc-provider cookies.keys is not required
  cookies: undefined,
};

const jwks_keys = process.env.OIDC_JWK_KEYS_FILE
  ? JSON.parse(fs.readFileSync(process.env.OIDC_JWK_KEYS_FILE!, "utf8"))
  : DEV_KEYS.jwks;

const cookies_keys = process.env.OIDC_COOKIE_KEYS_FILE
  ? JSON.parse(fs.readFileSync(process.env.OIDC_COOKIE_KEYS_FILE!, "utf8"))
  : DEV_KEYS.cookies;

export const oidc = new Provider(process.env.OIDC_ISSUER!, {
  adapter: PrismaAdapter,
  async findAccount(ctx, sub, token) {
    return {
      accountId: sub,
      async claims(use, scope, claims, rejected) {
        const hostname = new URL(sub).host;
        let user: IProfile = { sub };
        let instance: IInstance | undefined;

        try {
          const fetch = await APub.buildProfile(sub);
          if (fetch) user = fetch;
        } catch (e) {
          // it's aight, not required
        }

        try {
          instance = await getInstanceMeta(hostname);
        } catch (e) {
          // it's aight, not required
        }

        return { ...user, instance };
      },
    };
  },
  claims: {
    openid: ["sub", "name", "preferred_username", "profile", "picture"],
    instance: ["instance"],
  },
  jwks: {
    keys: jwks_keys,
  },
  cookies: {
    keys: cookies_keys,
  },
  pkce: { required: (ctx, client) => false },
  features: {
    devInteractions: { enabled: false },
    userinfo: { enabled: true },
    registration: {
      enabled: true,
      initialAccessToken: process.env.OIDC_REGISTRATION_TOKEN!,
    },
    registrationManagement: { enabled: true },
    revocation: { enabled: true },
    rpInitiatedLogout: {
      enabled: true,
      // logoutSource(ctx, form) {
      //   ctx.type = "html";
      //   ctx.body = "<h1>hi</h1> " + form;
      // },
      // postLogoutSuccessSource(ctx) {

      // }
    },
  },
  routes: {
    authorization: "/api/oidc/auth",
    backchannel_authentication: "/api/oidc/backchannel",
    code_verification: "/api/oidc/device",
    device_authorization: "/api/oidc/device/auth",
    end_session: "/api/oidc/logout",
    introspection: "/api/oidc/token/introspection",
    jwks: "/api/oidc/jwks",
    pushed_authorization_request: "/api/oidc/request",
    registration: "/api/oidc/registration",
    revocation: "/api/oidc/token/revoke",
    token: "/api/oidc/token",
    userinfo: "/api/oidc/me",
  },
});

type RouteName =
  | "authorization"
  | "backchannel_authentication"
  | "client_delete"
  | "client_update"
  | "client"
  | "code_verification"
  | "cors.device_authorization"
  | "cors.discovery"
  | "cors.introspection"
  | "cors.jwks"
  | "cors.pushed_authorization_request"
  | "cors.revocation"
  | "cors.token"
  | "cors.userinfo"
  | "device_authorization"
  | "device_resume"
  | "discovery"
  | "end_session_confirm"
  | "end_session_success"
  | "end_session"
  | "introspection"
  | "jwks"
  | "pushed_authorization_request"
  | "registration"
  | "resume"
  | "revocation"
  | "token"
  | "userinfo";

/**
 * Check if the requesting/requested user is banned via Shadow
 *
 * Requesting user: /api/oidc/auth
 * Requested user: /api/oidc/token,/api/oidc/me
 *
 * Requesting user should be given an UI explaining why, with an option to logout
 *
 * Requested user should return an access_token invalid error
 */
oidc.use(async (ctx, next) => {
  // check shadow privileges

  if (ctx.path.startsWith("/api/oidc/auth")) {
    // force all auth to show consent page
    ctx.query.prompt = "consent";
  }

  await next();

  const route: RouteName | undefined = ctx.oidc?.route;

  const BLACKLISTED_ENDPOINTS: RouteName[] = [
    "authorization",
    "backchannel_authentication",
    "code_verification",
    "token",
    "userinfo",
    "cors.token",
    "cors.userinfo",
  ];

  if (!route) return;

  if (BLACKLISTED_ENDPOINTS.indexOf(route) === -1) {
    // not an endpoint we care about
    return;
  }

  switch (route) {
    case "token":
    case "userinfo":
    case "cors.token":
    case "cors.userinfo":
      if ("access_token" in ctx.body) {
        // is an authorization response

        const token = await oidc.AccessToken.find(ctx.body.access_token);
        if (!token) {
          // no idea how we got here, but we can just let it be
          return;
        }

        const { accountId } = token;

        try {
          await ShadowAPI.canUserLogin({ sub: accountId, ip: ctx.ip });
        } catch (e) {
          await token.destroy(); // regardless of error, we destroy the access token

          if (e instanceof UserLoginError) {
            // has details we can share
            ctx.body = {
              error: "invalid_grant",
              error_description: "shadow: " + e.message,
            };
          } else {
            // internal shadow error
            ctx.body = {
              error: "invalid_grant",
              error_description: "internal error",
            };
          }
        }
      }
      break;
  }
});

/**
 * Check if instance supports OIDC discovery and dynamic client registration
 * @param instance_hostname
 * @returns
 */
export const doesInstanceSupportOIDC = async (instance_hostname: string) => {
  let issuer: Issuer;

  try {
    issuer = await Issuer.discover("https://" + instance_hostname);
  } catch (e) {
    return false;
  }

  if (typeof issuer.metadata.registration_endpoint === "undefined") {
    return false;
  }

  return true;
};
