/**
 * Utility functions for parsing NodeInfo
 *
 * @see https://nodeinfo.diaspora.software/protocol.html
 */

import {
  NODEINFO_SCHEMA_NAMESPACE,
  SUPPORTED_NODEINFO_VERSIONS,
  type NodeInfo,
} from "../types/nodeinfo.js";
import { safe_fetch } from "./fetch.js";

/**
 * Gets NodeInfo from hostname provided
 *
 * TODO: cache this
 *
 * 1. Read /.well-known/nodeinfo
 * 2. Read most up to date version known found
 *
 * @throws NodeInfo_Invalid
 * @throws NodeInfo_Unsupported
 * @param instance_hostname
 */
export const getNodeInfo = async (
  instance_hostname: string
): Promise<NodeInfo> => {
  const nodeinfo = await readWKNodeInfo(instance_hostname);
  return readNodeInfo(nodeinfo);
};

/**
 * Reads hostname's /.well-known/nodeinfo, validates & picks latest supported version
 *
 * This does not verify the full `Content-Type` header specified in the spec
 *
 * @see https://nodeinfo.diaspora.software/protocol.html
 * @param hostname
 * @returns Latest supported version's nodeinfo URL
 */
const readWKNodeInfo = async (hostname: string): Promise<string> => {
  const WK_URL = `https://${hostname}/.well-known/nodeinfo`;

  const req = await safe_fetch(WK_URL);

  if (req.status !== 200) {
    throw new NodeInfo_Invalid();
  }

  let data: any;
  try {
    data = await req.json();
  } catch (e) {
    throw new NodeInfo_Invalid();
  }

  // links that have the required props & types & are in the nodeinfo schema ns
  // sorted by ascii value in reverse order
  const validNodeInfoOptions: {
    rel: string;
    href: string;
  }[] = data?.links
    ?.filter(
      (link: any) =>
        typeof link.rel === "string" &&
        typeof link.href === "string" &&
        link.rel.indexOf(NODEINFO_SCHEMA_NAMESPACE) === 0
    )
    .map((a: any) => ({
      ...a,
      rel: a.rel.replace(NODEINFO_SCHEMA_NAMESPACE, ""),
    }))
    .sort((a: any, b: any) => b.rel - a.rel);

  let validVersion: string | undefined;
  for (let i = 0; i < validNodeInfoOptions.length; i++) {
    // this should return a version number
    let version = validNodeInfoOptions[i].rel;

    if (SUPPORTED_NODEINFO_VERSIONS.indexOf(version) > -1) {
      // this version is supported
      validVersion = version;
      break;
    }
  }

  if (!validVersion) throw new NodeInfo_Invalid();

  return validNodeInfoOptions.find((v) => v.rel === validVersion)!.href;
};

/**
 * Read the NodeInfo document specified
 *
 * @note This should be treated as user-supplied content
 * @note No properties are checked, everything is assumed to follow the spec
 *
 * @see https://nodeinfo.diaspora.software/schema.html
 * @param full_url
 */
const readNodeInfo = async <
  T extends NodeInfo["version"] = NodeInfo["version"]
>(
  full_url: string
): Promise<NodeInfo & { version: T }> => {
  const req = await safe_fetch(full_url);

  if (req.status !== 200) {
    throw new NodeInfo_Invalid();
  }

  let data: any;
  try {
    data = await req.json();
  } catch (e) {
    throw new NodeInfo_Invalid();
  }

  return data;
};

/****
 * Errors
 ****/

/**
 * Thrown when unable to parse a nodeinfo document
 */
class NodeInfo_Invalid extends Error {
  constructor() {
    super();
    this.name = "NodeInfo_Invalid";
  }
}

/**
 * Thrown when NodeInfo version found is unsupported
 */
class NodeInfo_Unsupported extends Error {
  constructor() {
    super();
    this.name = "NodeInfo_Unsupported";
  }
}
