import { safe_fetch } from "../fetch.js";
import { getSafeURL } from "../utils.js";

/**
 * Matches as close as possible to standard OpenID claims
 */
export interface IProfile {
  /**
   * URI of actor
   * @example https://grants.cafe/users/grant
   */
  sub: string;

  /**
   * Display name from AP actor
   */
  name?: string;

  /**
   * username@hostname.tld
   * @example grant@grants.cafe
   */
  preferred_username?: string;

  /**
   * HTML URL to profile page
   */
  profile?: string;

  /**
   * Raw URL to profile picture
   */
  raw_picture?: string;

  /**
   * URL to profile picture
   */
  picture?: string;
}

export const getUserMeta = async (
  user: [username: string, hostname: string]
): Promise<IProfile> => {
  const req = await safe_fetch(
    `https://${user[1]}/.well-known/webfinger?resource=acct:${user.join("@")}`
  );

  if (req.status !== 200) {
    // will throw if not found
    throw new Error();
  }

  let data: any;
  try {
    data = await req.json();
  } catch (e) {
    throw new Error();
  }

  const getLinkFor = (rel: string, type: string): string | undefined => {
    const link = data?.links?.find(
      (l: any) =>
        typeof l.rel === "string" &&
        typeof l.type === "string" &&
        l.rel === rel &&
        l.type === type
    );
    if (!link || typeof link.href !== "string") return undefined;

    return link.href;
  };

  const apURL = getLinkFor("self", "application/activity+json");
  const profilePage =
    getLinkFor("http://webfinger.net/rel/profile-page", "text/html") || apURL;

  if (!apURL) {
    // url is not found, shouldn't be a valid user
    throw new Error();
  }

  const apReq = await safe_fetch(apURL, {
    headers: {
      Accept: "application/activity+json",
    },
  });

  if (apReq.status !== 200) {
    throw new Error();
  }

  let apData: any;
  try {
    apData = await apReq.json();
  } catch (e) {
    throw new Error();
  }

  return {
    sub: user.join("@"),
    name: apData.name,
    raw_picture: apData.icon?.url,
    picture: getSafeURL(apData.icon?.url),
    preferred_username: apData.preferredUsername,
    profile: profilePage,
  };
};
