/**
 * Utility functions for getting instance-specific metadata
 */

import { NodeInfo } from "../../types/nodeinfo.js";
import { safe_fetch } from "../fetch.js";
import { getNodeInfo } from "../nodeinfo.js";
import { getSafeURL } from "../utils.js";

export interface IInstance {
  software: {
    name: string;
    version: string;
    logo_uri?: string;
    repository?: string;
    homepage?: string;
  };
  instance: {
    /**
     * Untrusted URL
     */
    raw_logo_uri?: string;
    /**
     * Untrusted URL
     */
    raw_banner_uri?: string;
    logo_uri?: string;
    banner_uri?: string;
    name?: string;
  };
}

/**
 * Get instance metadata from hostname
 *
 * TODO: write software#logo_uri
 *
 * @throws NodeInfo_Invalid|NodeInfo_Unsupported if nodeinfo param is not provided
 * @param instance_hostname
 * @param nodeinfo
 */
export const getInstanceMeta = async (
  instance_hostname: string,
  nodeinfo?: NodeInfo
): Promise<IInstance> => {
  let _nodeinfo: NodeInfo | undefined = nodeinfo;

  if (!_nodeinfo) {
    _nodeinfo = await getNodeInfo(instance_hostname);
  }

  let software: IInstance["software"] = {
    name: _nodeinfo.software.name,
    version: _nodeinfo.software.version,
    ...("repository" in _nodeinfo.software && {
      repository: _nodeinfo.software.repository,
    }),
    ...("homepage" in _nodeinfo.software && {
      homepage: _nodeinfo.software.homepage,
    }),
  };

  try {
    const fedidbReq = await safe_fetch(
      `https://api.fedidb.org/v1/software/${software.name}`
    );
    if (fedidbReq.status !== 200) throw new Error();

    const fedidbRes: any = await fedidbReq.json();
    if (!fedidbRes) throw new Error();

    software.logo_uri =
      typeof fedidbRes.logo_url === "string" && fedidbRes.logo_url;
  } catch (e) {
    // ignore failed
  }

  let instance: IInstance["instance"] = {};

  try {
    switch (_nodeinfo.software.name) {
      case "mastodon": {
        const metaReq = await safe_fetch(
          `https://${instance_hostname}/api/v2/instance`
        );
        if (metaReq.status !== 200) throw new Error();

        const metaRes: any = await metaReq.json();

        if (!metaRes) throw new Error();

        instance.name = typeof metaRes.title === "string" && metaRes.title;
        instance.raw_banner_uri =
          typeof metaRes?.thumbnail?.url === "string" && metaRes.thumbnail.url;
        break;
      }
      case "lemmy": {
        const metaReq = await safe_fetch(
          `https://${instance_hostname}/api/v3/site`
        );
        if (metaReq.status !== 200) throw new Error();

        const metaRes: any = await metaReq.json();

        if (!metaRes) throw new Error();

        instance.name =
          typeof metaRes.site_view?.site?.name === "string" &&
          metaRes.site_view.site.name;
        instance.raw_logo_uri =
          typeof metaRes.site_view?.site?.icon === "string" &&
          metaRes.site_view.site.icon;
        instance.raw_banner_uri =
          typeof metaRes.site_view?.site?.banner === "string" &&
          metaRes.site_view.site.banner;
        break;
      }
    }
  } catch (e) {
    // ignore meta if failed
  }

  if (instance.raw_banner_uri)
    instance.banner_uri = getSafeURL(instance.raw_banner_uri);
  if (instance.raw_logo_uri)
    instance.logo_uri = getSafeURL(instance.raw_logo_uri);

  return {
    software,
    instance,
  };
};
