import express from "express";
import session from "express-session";
import cors from "cors";
import bodyParser from "body-parser";
import path from "path";
import { oidc } from "./oidc.js";
import { makeClientPublic } from "./utils.js";
import { errors as OIDC_Errors } from "oidc-provider";

import "../types/session-types.js";
import { APIRouter } from "./api.js";
import { integrateFederation } from "@fedify/express";
import { federation, USER_IDENTIFIER } from "./apub/federation.js";
import { APub } from "./apub/utils.js";
import { APIAdminRouter } from "./api_admin.js";
import { Handoff } from "../handoff/index.js";
import { DocLinks } from "./DocLinks.js";

export const app = express();

if (process.env.NODE_ENV === "production") app.set("trust proxy", 1);

if (process.env.NODE_ENV === "development" && process.env.DEV_TRUST_PROXIES) {
  console.log("DEV_TRUST_PROXIES enabled, trusting proxies");
  app.set("trust proxy", 1);
  oidc.proxy = true;
  app.enable("trust proxy");
}

app.get("/logout", (req, res) => res.redirect("/api/oidc/logout"));
app.all("/api/oidc/*", oidc.callback());
app.get("/.well-known/openid-configuration", oidc.callback());
app.use(bodyParser.json());

app.use(
  session({
    secret: process.env.SESSION_SECRET!,
    resave: false,
    saveUninitialized: true,
    cookie: {
      secure:
        process.env.NODE_ENV === "production" && !process.env.USE_INSECURE,
      sameSite: "lax",
      httpOnly: false,
    },
    // TODO: do not use memory store
  })
);

app.get("/.well-known/com.sc07.fediverse-auth", (req, res) => {
  res.json({
    registration: {
      mode: process.env.OIDC_REGISTRATION_TOKEN ? "private" : "public",
    },
    fediverse: {
      account: APub.accountHandle,
    },
    handoff: Handoff.canEnable()
      ? { enabled: true, token: Handoff.HANDOFF_TOKEN }
      : { enabled: false },
    _meta: {
      source: DocLinks.SOURCE,
      docs: DocLinks.DOCS,
    },
  });
});

try {
  app.use("/handoff", Handoff.get().router);
} catch (e) {
  console.warn("Failed to activate Handoff:", (e as any)?.message || e);
}

const interactionMiddleware = (
  req: express.Request,
  resp: express.Response
) => {
  return new Promise<
    | { type: "continue" }
    | { type: "redirect"; to: string }
    | { type: "error"; error: "session_lost" }
    | { type: "error"; error: "unknown" }
  >(async (res) => {
    const interaction = await oidc.Interaction.find(req.params.uid);
    if (interaction?.prompt.name === "login") {
      if (typeof req.session.user === "undefined") {
        res({
          type: "redirect",
          to:
            "/login?return=" +
            encodeURIComponent("/interaction/" + req.params.uid),
        });
      } else {
        try {
          const actor = await APub.lookupActor(req.session.user.sub);
          if (!actor)
            throw new Error(
              "Actor " + req.session.user.sub + " is not an actor"
            );

          const returnTo = await oidc.interactionResult(req, resp, {
            login: { accountId: actor.id!.toString() },
          });

          req.session.destroy(() => {
            res({ type: "redirect", to: returnTo });
          });
        } catch (e) {
          console.error("Error while in interaction middleware", e);

          req.session.destroy(() => {
            if (e instanceof OIDC_Errors.SessionNotFound) {
              res({ type: "error", error: "session_lost" });
            } else {
              res({ type: "error", error: "unknown" });
            }
          });
        }
      }
    } else {
      res({ type: "continue" });
    }
  });
};

if (process.env.NODE_ENV === "development") {
  // expose the internals of the interaction middleware for the vite dev server to access
  app.post("/_dev/interaction/:uid", async (req, res) => {
    const middleware = await interactionMiddleware(req, res);

    switch (middleware.type) {
      case "redirect":
        res.redirect(middleware.to);
        break;
      case "error":
        switch (middleware.error) {
          case "session_lost":
            res.send("<h1>session lost</h1><p>Try login again</p>");
            break;
          case "unknown":
            res.send("<h1>unknown error</h1>");
            break;
        }
        break;
      case "continue":
      default:
        res.end();
        break;
    }
  });
}

app.use("/interaction/:uid", async (req, res, next) => {
  const middleware = await interactionMiddleware(req, res);

  switch (middleware.type) {
    case "redirect":
      res.redirect(middleware.to);
      break;
    case "error":
      switch (middleware.error) {
        case "session_lost":
          res.send("<h1>session lost</h1><p>Try login again</p>");
          break;
        case "unknown":
          res.send("<h1>unknown error</h1>");
          break;
      }
      break;
    case "continue":
    default:
      next();
      break;
  }
});

if (process.env.SERVE_FRONTEND) {
  const indexFile = path.join(process.env.SERVE_FRONTEND, "index.html");

  app.use(express.static(process.env.SERVE_FRONTEND));
  app.get(["/", "/interaction*", "/login"], (req, res) => {
    res.sendFile(indexFile);
  });
} else {
  app.get("/", (req, res) => {
    res.send("fediverse-auth");
  });
}

app.use("/api/v1", APIRouter);
if (process.env.ADMIN_TOKEN && process.env.ADMIN_TOKEN.length > 1)
  app.use("/api/admin", APIAdminRouter);

app.use(integrateFederation(federation, (req) => {}));
