import {
  ChatMessage,
  Create,
  createFederation,
  Endpoints,
  exportJwk,
  Follow,
  generateCryptoKeyPair,
  importJwk,
  isActor,
  lookupObject,
  Note,
  Reject,
  Service,
} from "@fedify/fedify";
import { RedisKvStore, RedisMessageQueue } from "@fedify/redis";
import { Redis } from "ioredis";
import { FedifyTransformers } from "./transformers.js";
import { prisma } from "../prisma.js";
import { APub } from "./utils.js";
import { Handoff } from "../../handoff/index.js";

// @auth@some.domain
export const USER_IDENTIFIER = "auth";

// CryptoKeyPair is from the Web Crypto API
type CryptoKeyPair = Awaited<ReturnType<typeof generateCryptoKeyPair>>;

export const federation = createFederation<void>({
  kv: new RedisKvStore(new Redis(process.env.REDIS_URI)),
  queue: new RedisMessageQueue(() => new Redis(process.env.REDIS_URI)),
  activityTransformers: FedifyTransformers,
  manuallyStartQueue: true,
  origin: process.env.OIDC_ISSUER,
  userAgent: {
    software: "FediverseAuth/" + (process.env.VERSION || "dev"),
    url: process.env.OIDC_ISSUER,
  },

  allowPrivateAddress: process.env.NODE_ENV === "development",
  skipSignatureVerification: process.env.NODE_ENV === "development",
});

federation
  .setActorDispatcher("/x/users/{identifier}", async (ctx, identifier) => {
    if (identifier !== USER_IDENTIFIER) return null;

    return new Service({
      id: ctx.getActorUri(USER_IDENTIFIER),
      name: process.env.SERVICE_NAME,
      summary: "//TODO",
      manuallyApprovesFollowers: true,
      preferredUsername: USER_IDENTIFIER,
      url: ctx.getActorUri(USER_IDENTIFIER),
      inbox: ctx.getInboxUri(USER_IDENTIFIER),
      outbox: ctx.getOutboxUri(USER_IDENTIFIER),
      endpoints: new Endpoints({
        sharedInbox: ctx.getInboxUri(),
      }),
      publicKeys: (await ctx.getActorKeyPairs(USER_IDENTIFIER)).map(
        (keyPair) => keyPair.cryptographicKey
      ),
    });
  })
  .setKeyPairsDispatcher(async (ctx, identifier) => {
    if (identifier !== USER_IDENTIFIER) return [];

    const result: CryptoKeyPair[] = [];

    const rsaPair = await prisma.fediverseKeyPair.findFirst({
      where: {
        keyType: "RSASSA-PKCS1-v1_5",
      },
    });
    if (rsaPair) {
      const { privateKey, publicKey } = JSON.parse(rsaPair.value);
      result.push({
        privateKey: await importJwk(privateKey, "private"),
        publicKey: await importJwk(publicKey, "public"),
      });
    } else {
      const { privateKey, publicKey } = await generateCryptoKeyPair(
        "RSASSA-PKCS1-v1_5"
      );
      await prisma.fediverseKeyPair.create({
        data: {
          keyType: "RSASSA-PKCS1-v1_5",
          value: JSON.stringify({
            privateKey: await exportJwk(privateKey),
            publicKey: await exportJwk(publicKey),
          }),
        },
      });
      result.push({
        privateKey,
        publicKey,
      });
    }

    // TODO: add ed25519 back to response as the library complains if it isn't returned in some circumstances
    // Lemmy will reject user resolves if the public_key field isn't exactly one key response
    // @see https://github.com/LemmyNet/lemmy/blob/ed5a3831aa3ac10c9e0de6b70a7df282c94fcdb3/crates/apub/src/protocol/objects/person.rs#L33

    return result;
  });

federation
  .setInboxListeners("/x/users/{identifier}/inbox", "/inbox")
  .on(Follow, async (ctx, follow) => {
    if (follow.id == null || follow.actorId == null || follow.objectId == null)
      return;

    const parsed = ctx.parseUri(follow.objectId);
    if (parsed?.type !== "actor" || parsed.identifier !== USER_IDENTIFIER)
      return;

    const follower = await follow.getActor(ctx);
    if (follower == null) return;

    // reject incoming follow requests
    await ctx.sendActivity(
      { identifier: parsed.identifier },
      follower,
      new Reject({
        actor: follow.objectId,
        object: follow,
      })
    );
  })
  .on(Create, async (ctx, create) => {
    const actor = await create.getActor(ctx);
    const object = await create.getObject(ctx);

    if (!actor || !object) {
      console.log("create object or actor didn't exist", {
        create,
        actor,
        object,
      });
      return;
    }

    if (object instanceof Note || object instanceof ChatMessage) {
      Handoff.get().activitypub.handle(actor, object, create);
    } else {
      console.log("create object unknown type", create, object);
    }
  });

federation.setOutboxDispatcher(
  "/x/users/{identifier}/outbox",
  (ctx, identifier) => {
    return {
      items: [],
    };
  }
);

federation.setObjectDispatcher(
  ChatMessage,
  "/x/object/chatmessage/{id}",
  async (ctx, { id }) => {
    const authSession = await prisma.authSession.findFirst({
      where: {
        objectId: id,
      },
    });

    if (!authSession) return null;

    const recipient = await lookupObject(
      authSession.user_sub,
      APub.options(ctx)
    );
    const apub = new APub(ctx);

    if (!isActor(recipient)) return null;

    return apub.build("ChatMessage", {
      ...authSession,
      target: recipient!,
    });
  }
);

federation.setObjectDispatcher(
  Note,
  "/x/object/note/{id}",
  async (ctx, { id }) => {
    const authSession = await prisma.authSession.findFirst({
      where: {
        objectId: id,
      },
    });

    if (!authSession) return null;

    const recipient = await lookupObject(
      authSession.user_sub,
      APub.options(ctx)
    );
    const apub = new APub(ctx);

    if (!isActor(recipient)) return null;

    return apub.build("Note", {
      ...authSession,
      target: recipient!,
    });
  }
);
