import express from "express";
import cookieParser from "cookie-parser";
import { doesInstanceSupportOIDC, oidc } from "./oidc.js";
import {
  DOMAIN_REGEX,
  getExpressIP,
  isInstanceDomainValid,
  makeClientPublic,
} from "./utils.js";
import { getNodeInfo } from "./nodeinfo.js";
import { prisma } from "./prisma.js";
import { IProfile, getUserMeta } from "./instance/userMeta.js";
import { IInstance, getInstanceMeta } from "./instance/instanceMeta.js";
import { ShadowAPI, UserLoginError } from "./shadow.js";
import { APub } from "./apub/utils.js";
import { AuthSession } from "../controllers/AuthSession.js";

const app = express.Router();

app.use(cookieParser());

/****
 * Fedi auth & account verification
 ****/

/**
 * Get the current user's session if it exists
 */
app.get("/whoami", async (req, res) => {
  const ctx = oidc.app.createContext(req, res);
  const session = await oidc.Session.get(ctx);
  if (!session || !session.accountId)
    return res.json({ success: false, error: "no-session" });

  const hostname = session.accountId.split("@")[1];
  let user: IProfile = { sub: session.accountId };
  let instance: IInstance | undefined;

  try {
    const fetch = await APub.buildProfile(user.sub);
    if (fetch) user = fetch;
  } catch (e) {
    // it's aight, not required
  }

  try {
    instance = await getInstanceMeta(hostname);
  } catch (e) {
    // it's aight, not required
  }

  res.json({
    handle: user.preferred_username,
    ...user,
    instance,
  });
});

/*
  # login steps #
  - each login step saves progress in the session
  - *similar* to matrix's auth
  - when a step returns {success:true}, that step is completed and the next step is sent back to the client
*/

/**
 * Step #1: Verify instance
 * Check if an instance is supported
 *
 * TODO: cache this response (or parts of it)
 */
app.post("/login/step/instance", async (req, res) => {
  let domain: string;

  if (typeof req.body.domain !== "string") {
    return res.status(400).json({
      success: false,
      error: "Domain is not a string",
    });
  }

  domain = req.body.domain;

  if (!new RegExp(DOMAIN_REGEX).test(domain) || domain.indexOf(".") === -1) {
    return res.status(400).json({
      success: false,
      error: `The domain proided (${domain}) is not a valid domain`,
    });
  }

  if (!(await isInstanceDomainValid(domain))) {
    return res.status(400).json({
      success: false,
      error:
        "The instance provided either doesn't have a valid NodeInfo or doesn't support ActivityPub",
    });
  }

  const nodeinfo = await getNodeInfo(domain);

  req.session.login = {
    prompt: "USERNAME", // change this if oidc is available
    instance: domain,
    attempt: 0,
  };

  // const oidcSupport = await doesInstanceSupportOIDC(domain);

  // TODO: detect next step for the client (oidc, receive code, send code)

  req.session.save(() => {
    res.send({ success: true, step: "USERNAME" });
  });
});

app.post("/login/step/username", async (req, res) => {
  if (!req.session?.login || req.session.login.prompt !== "USERNAME") {
    return res.status(400).json({ success: false, error: "wrong_step" });
  }

  const { instance } = req.session.login;

  let username: string;
  if (typeof req.body.username !== "string") {
    return res.status(400).json({
      success: false,
      error: "username is not a string",
    });
  }
  username = req.body.username.toLowerCase();

  req.session.login.username = username;
  // this is the prompt for the user
  req.session.login.prompt = "ENTER_CODE";

  const existing = await AuthSession.getActive(`${username}@${instance}`);

  if (existing) {
    // if there's an active session, don't create another one
    req.session.login.session_id = existing.id;
    req.session.save(() => {
      res.send({
        success: true,
        data: {
          session_id: existing.id,
          account: APub.accountHandle,
        },
      });
    });
    return;
  }

  const session = await AuthSession.create(`${username}@${instance}`);
  req.session.login.session_id = session.id;

  try {
    await APub.sendDM(session);

    req.session.save(() => {
      res.send({
        success: true,
        data: {
          session_id: session.id,
          account: APub.accountHandle,
        },
      });
    });
  } catch (e) {
    console.error(
      "Error while delivering to " + [username, instance].join("@"),
      e
    );

    await prisma.authSession.delete({ where: { id: session.id } });
    req.session.login.session_id = undefined;

    req.session.save(() => {
      res.send({
        success: false,
        error:
          "Error while sending: " + ((e as any)?.message || "unknown error"),
      });
    });
  }
});

app.post("/login/step/verify", async (req, res) => {
  if (
    !req.session?.login ||
    ["ENTER_CODE", "SEND_CODE"].indexOf(req.session.login.prompt) === -1
  ) {
    return res.status(400).json({ success: false, error: "wrong_step" });
  }

  const { session_id, username, instance } = req.session.login;

  if (req.session.login.attempt > 5) {
    req.session.destroy(() => {
      res.status(400).json({ success: false, error: "too_many_attempts" });
    });
    return;
  }

  const session = await prisma.authSession.findFirst({
    where: {
      id: session_id,
      user_sub: [username, instance].join("@"),
    },
  });

  if (!session) {
    req.session.login = undefined;
    req.session.save(() => {
      res.status(400).json({ success: false, error: "session_lost" });
    });
    return;
  }

  if (process.env.BYPASS_CODE_VERIFICATION) {
    console.warn("BYPASS_CODE_VERIFICATION is set; not verifying 2fa codes");

    const user = await APub.lookupActor(session.user_sub);
    if (!user)
      throw new Error("BYPASS_CODE_VERIFICATION user is not an actor!");

    req.session.user = {
      sub: user.id!.toString(),
      handle: session.user_sub as any,
    };
    req.session.login = undefined;
    req.session.save(() => {
      res.json({ success: true });
    });
    return;
  }

  // code has been sent, we're expecting the user to give a code
  let code: string;

  if (typeof req.body.code !== "string") {
    return res
      .status(400)
      .json({ success: false, error: "code is not a string" });
  }
  code = req.body.code;

  if (session.one_time_code !== code) {
    req.session.login.attempt++;
    req.session.save(() => {
      res.status(400).json({ success: false, error: "code_invalid" });
    });
    return;
  }

  const user = await APub.lookupActor(session.user_sub);
  if (!user) throw new Error("Code verification: user is not an actor!");

  await prisma.authSession.delete({ where: { id: session.id } });
  req.session.user = {
    sub: user.id!.toString(),
    handle: session.user_sub as any,
  };
  req.session.login = undefined;
  req.session.save(() => {
    res.json({ success: true });
  });
});

/****
 * API methods for getting oidc-provider metadata for the client-side render
 ****/

app.get("/interaction", async (req, res) => {
  const { id } = req.query;
  if (typeof id !== "string") {
    return res
      .status(400)
      .json({ success: false, error: "Missing interaction ?id" });
  }

  const interaction = await oidc.Interaction.find(id);
  if (!interaction) {
    return res
      .status(404)
      .json({ success: false, error: "Unknown interaction" });
  }

  res.json(interaction);
});

app.get("/client", async (req, res) => {
  const { id } = req.query;
  if (typeof id !== "string") {
    return res
      .status(400)
      .json({ success: false, error: "Missing client ?id" });
  }

  const client = await oidc.Client.find(id);
  if (!client) {
    return res.status(404).json({ success: false, error: "Unknown client" });
  }

  res.json(makeClientPublic(client));
});

// TODO: maybe add a security cookie?
app.post("/interaction/:uid/confirm", async (req, res) => {
  try {
    const interaction = await oidc.Interaction.find(req.params.uid);
    if (!interaction)
      return res
        .status(400)
        .json({ success: false, error: "Unknown interaction" });

    if (interaction.prompt.name !== "consent")
      return res
        .status(400)
        .json({ success: false, error: "Invalid interaction" });

    if (!interaction.session?.accountId) {
      return res
        .status(400)
        .json({ success: false, error: "Failed to get accountId" });
    }

    const ipAddress = getExpressIP(req);

    try {
      await ShadowAPI.canUserLogin({
        sub: interaction.session.accountId,
        ip: ipAddress,
      });
    } catch (e) {
      if (e instanceof UserLoginError) {
        res.status(400).json({
          success: false,
          error: "shadow",
          metadata: {
            message: e.message,
          },
        });
      } else {
        res.status(400).json({
          success: false,
          error: "shadow",
          metadata: {
            message: "internal error",
          },
        });
      }
      return;
    }

    let grant;
    if (interaction.grantId) {
      grant = await oidc.Grant.find(interaction.grantId);
    } else {
      grant = new oidc.Grant({
        accountId: interaction.session?.accountId,
        clientId: interaction.params.client_id as string,
      });
    }

    if (interaction.prompt.details.missingOIDCScope) {
      grant!.addOIDCScope(
        (interaction.prompt.details.missingOIDCScope as any).join(" ")
      );
    }

    let grantId = await grant!.save();

    let consent: any = {};
    if (!interaction.grantId) {
      // we're just modifying an existing one
      consent.grantId = grantId;
    }

    // merge with last submission
    interaction.result = { ...interaction.lastSubmission, consent };
    await interaction.save(interaction.exp - Math.floor(Date.now() / 1000));

    res.json({
      success: true,
      returnTo: interaction.returnTo,
    });
  } catch (e) {
    console.error(req.originalUrl, e);
    res.status(500).json({
      success: false,
      error: "Internal server error",
    });
  }
});

// TODO: maybe add a security cookie?
app.post("/interaction/:uid/abort", async (req, res) => {
  try {
    const interaction = await oidc.Interaction.find(req.params.uid);
    if (!interaction)
      return res
        .status(400)
        .json({ success: false, error: "Unknown interaction" });

    interaction.result = {
      error: "access_denied",
      error_description: "End-User aborted interaction",
    };
    await interaction.save(interaction.exp - Math.floor(Date.now() / 1000));

    res.json({
      success: true,
      returnTo: interaction.returnTo,
    });
  } catch (e) {
    console.error(req.originalUrl, e);
    res.status(500).json({
      success: false,
      error: "Internal server error",
    });
  }
});

export { app as APIRouter };
