import { CronJob } from "cron";
import { AuthSession } from "../controllers/AuthSession.js";
import { APub } from "../lib/apub/utils.js";
import { prisma } from "../lib/prisma.js";

export class Jobs {
  private static instance: Jobs;

  static start() {
    this.instance = new Jobs();

    CronJob.from({
      name: "Destroy expired tokens",
      cronTime: "*/5 * * * *",
      onTick: this.instance.handleExpiredTokens.bind(this),
      start: true,
    });

    CronJob.from({
      name: "Fediverse fetch job",
      cronTime: "*/5 * * * *",
      onTick: this.instance.handleBatchFetch.bind(this),
      start: true,
    });
  }

  async handleExpiredTokens() {
    const expired = await AuthSession.getExpired();
    console.debug(`Deleting ${expired.length} expired tokens`);

    for (const session of expired) {
      console.debug(`Deleting ${session.id}`);
      await APub.deleteDM(session);
      await prisma.authSession.delete({ where: { id: session.id } });
    }
  }

  async handleBatchFetch() {
    const handlesToFetch = await prisma.fediverseUser.findMany({
      where: {
        objectId: null,
        error: false,
      },
      take: 20,
    });
    const total = await prisma.fediverseUser.count({
      where: {
        objectId: null,
        error: false,
      },
    });

    if (handlesToFetch.length > 0) {
      console.log(`Fetching ${handlesToFetch.length}/${total} handles...`);

      for (const user of handlesToFetch) {
        await APub.lookupActor("@" + user.handle)
          .then((resource) => {
            const uri = resource?.id;

            if (uri) {
              return prisma.fediverseUser.update({
                where: {
                  id: user.id,
                },
                data: {
                  objectId: uri.toString(),
                },
              });
            } else {
              throw new Error("Webfinger doesn't have activity+json object");
            }
          })
          .catch((e) => {
            console.error(`Failed to fetch ${user.handle}`, e);
            return prisma.fediverseUser.update({
              where: {
                id: user.id,
              },
              data: {
                error: true,
              },
            });
          });
      }

      console.log(`Finished fetching handles in batch`);
    }
  }
}
