import { configure, getConsoleSink } from "@logtape/logtape";
import { app as Express } from "./lib/express.js";
import { oidc } from "./lib/oidc.js";
import { FederationWorker } from "./lib/apub/worker.js";
import { Jobs } from "./jobs/jobs.js";

if (typeof process.env.SESSION_SECRET !== "string") {
  throw new Error("SESSION_SECRET is not defined");
}

if (typeof process.env.NODE_ENV !== "string") {
  process.env.NODE_ENV = "development";
}

if (typeof process.env.OIDC_ISSUER !== "string") {
  throw new Error("OIDC_ISSUER is not defined");
}

if (typeof process.env.OIDC_REGISTRATION_TOKEN !== "string") {
  // throw new Error("OIDC_REGISTRATION_TOKEN is not defined");
  console.warn(
    "OIDC_REGISTRATION_TOKEN is not defined\nThis removes the requirement for passing a token and anyone can then register a client"
  );
}

if (process.env.NODE_ENV === "production") {
  if (typeof process.env.SESSION_SECRET !== "string") {
    throw new Error("SESSION_SECRET is not defined");
  }

  if (typeof process.env.OIDC_JWK_KEYS_FILE !== "string") {
    throw new Error("OIDC_JWK_KEYS_FILE is not defined");
  }

  if (typeof process.env.OIDC_COOKIE_KEYS_FILE !== "string") {
    throw new Error("OIDC_COOKIE_KEYS_FILE is not defined");
  }

  if (!process.env.USE_INSECURE) {
    oidc.proxy = true;
    Express.enable("trust proxy");
  }
}

await configure({
  sinks: { console: getConsoleSink() },
  loggers: [
    {
      category: "fedify",
      sinks: ["console"],
      lowestLevel: "debug",
    },
  ],
  filters: {},
});

/**
 * Should the worker scripts run now?
 *
 * Any of:
 * - NODE_ENV = development
 * - NODE_TYPE = worker
 */
const shouldStartWorker =
  process.env.NODE_ENV === "development" || process.env.NODE_TYPE === "worker";
/**
 * Should the main scripts run now?
 *
 * Any of:
 * - NODE_ENV = development
 * - NODE_ENV = production & NODE_TYPE != worker
 */
const shouldStartMain =
  process.env.NODE_ENV === "development" ||
  (process.env.NODE_ENV === "production" && process.env.NODE_TYPE !== "worker");

console.log("\n");
console.log("Initializing fediverse-auth...", {
  "Start Main": shouldStartMain,
  "Start Worker": shouldStartWorker,
});
console.log("\n");

if (shouldStartWorker) {
  Jobs.start();
  void FederationWorker.create();
}

if (shouldStartMain) {
  Express.listen(process.env.PORT, () => {
    console.log("Listening on :" + process.env.PORT);
  });
}
