import { Actor, ChatMessage, Create, Mention, Note } from "@fedify/fedify";
import { HandoffSession } from "../controllers/HandoffSession.js";
import { APub } from "../lib/apub/utils.js";
import { USER_IDENTIFIER } from "../lib/apub/federation.js";
import { Temporal } from "@js-temporal/polyfill";
import { stripHtml } from "string-strip-html";

type Message = Note | ChatMessage;

/**
 * Handle automated exchange of tokens from third party apps
 */
export class HandoffActivityPub {
  async handle(actor: Actor, object: Message, create: Create) {
    if (!object.content && !object.contents[0]) return;
    if (!actor.id) return;

    const check = this.isObjectValid(object);
    if (!check.valid) return;

    const handoff = await HandoffSession.create(actor.id.toString());
    const apub = APub.get();
    const sender = apub.ctx.getActorUri(USER_IDENTIFIER);

    if (object instanceof Note) {
      await apub.sendNote(
        "handoff-" + handoff.id,
        actor,
        new Note({
          id: apub.ctx.getObjectUri(Note, { id: "handoff-" + handoff.id }),
          attribution: sender,
          to: actor.id,
          published: Temporal.Instant.from(handoff.createdAt.toISOString()),
          replyTarget: object,
          content: handoff.getURL().toString(),
          tags: [
            new Mention({
              href: actor.id,
              name: actor.id!.toString(),
            }),
          ],
        })
      );
    }

    if (object instanceof ChatMessage) {
      await apub.sendChatMessage(
        "handoff-" + handoff.id,
        actor,
        new ChatMessage({
          id: apub.ctx.getObjectUri(ChatMessage, {
            id: "handoff-" + handoff.id,
          }),
          attribution: sender,
          to: actor.id,
          published: Temporal.Instant.from(handoff.createdAt.toISOString()),
          replyTarget: object,
          content: handoff.getURL().toString(),
        })
      );
    }
  }

  isObjectValid(
    object: Message
  ): { valid: false } | { valid: true; software: string } {
    const TOKEN = process.env.AUTOMATED_EXCHANGE_TOKEN;

    if (!TOKEN) return { valid: false };

    const content = stripHtml(
      object.content?.toString() || object.contents[0]?.toString()
    ).result;

    if (content.indexOf(TOKEN) === -1) return { valid: false };

    return {
      valid: true,
      software: "unknown",
    };
  }
}
