import { AuthSession as DBAuthSession } from "@prisma/client";
import { prisma } from "../lib/prisma.js";

export class AuthSession {
  static generateCode(): string {
    return "."
      .repeat(5)
      .split("")
      .map(() => Math.floor(Math.random() * 10))
      .join("");
  }
  static async create(handle: `${string}@${string}`): Promise<AuthSession> {
    // day in seconds
    const expiresAt = new Date(Date.now() + 1000 * 60 * 60 * 24);

    const session = await prisma.authSession.create({
      data: {
        one_time_code: this.generateCode(),
        user_sub: handle,
        expiresAt,
      },
    });

    return new AuthSession(session);
  }
  static async getExpired() {
    const sessions = await prisma.authSession.findMany({
      where: {
        expiresAt: {
          lte: new Date(),
        },
      },
    });
    return sessions.map((d) => new AuthSession(d));
  }
  static async getActive(
    handle: `${string}@${string}`
  ): Promise<AuthSession | null> {
    const session = await prisma.authSession.findFirst({
      where: {
        user_sub: handle,
        expiresAt: {
          gt: new Date(),
        },
      },
    });

    if (!session) return null;

    return new AuthSession(session);
  }

  private _id: string;
  private _user_sub: `${string}@${string}`;
  private _createdAt: Date;
  private _expiresAt: Date;
  private _code: string;

  private constructor(session: DBAuthSession) {
    this._id = session.id;
    this._user_sub = session.user_sub as any;
    this._createdAt = session.createdAt;
    this._expiresAt = session.expiresAt;
    this._code = session.one_time_code;
  }

  get id() {
    return this._id;
  }

  get user_sub() {
    return this._user_sub;
  }

  get createdAt() {
    return this._createdAt;
  }

  get expiresAt() {
    return this._expiresAt;
  }

  get code() {
    return this._code;
  }
}
