import * as openid from "openid-client";

const { AUTH_ENDPOINT, AUTH_CLIENT, AUTH_SECRET, AUTH_CALLBACK } = process.env;

const HOST =
  new URL(AUTH_CALLBACK!).protocol + "//" + new URL(AUTH_CALLBACK!).host;

export namespace OpenID {
  export type ExchangeToken = Awaited<
    ReturnType<typeof openid.authorizationCodeGrant>
  >;
}

/**
 * Originates from Canvas
 * @see https://sc07.dev/sc07/canvas/-/blob/dc2764d85049ae9c6b890f0b50386a2a36048dd8/packages/server/src/controllers/OpenIDController.ts
 */
export class OpenIDController {
  private static instance: OpenIDController | undefined;
  config: openid.Configuration = {} as any;

  private constructor() {}

  static async initialize() {
    if (typeof OpenIDController.instance !== "undefined") {
      throw new Error(
        "OpenIDController#instance called when already initialized"
      );
    }

    const instance = (OpenIDController.instance = new OpenIDController());

    instance.config = await openid.discovery(
      new URL(AUTH_ENDPOINT!),
      AUTH_CLIENT!,
      {
        client_secret: AUTH_SECRET,
      }
    );
  }

  static get(): OpenIDController {
    if (typeof OpenIDController.instance === "undefined") {
      throw new Error("OpenIDController#get called when not initialized");
    }

    return OpenIDController.instance;
  }

  getRedirectUrl() {
    return AUTH_CALLBACK!;
  }

  getAuthorizationURL() {
    return openid
      .buildAuthorizationUrl(this.config, {
        redirect_uri: this.getRedirectUrl(),
        prompt: "consent",
        scope: "openid instance",
      })
      .toString();
  }

  exchangeToken(relativePath: string) {
    return openid.authorizationCodeGrant(
      this.config,
      new URL(relativePath, HOST)
    );
  }

  userInfo<Data extends object = object>(
    accessToken: string,
    expectedSub: string
  ): Promise<openid.UserInfoResponse & Data> {
    return openid.fetchUserInfo(this.config, accessToken, expectedSub) as any;
  }
}
