import express from "express";
import { OpenIDController, OpenID } from "./oidc";
import { ResponseBodyError } from "openid-client";
import { stripHtml } from "string-strip-html";

OpenIDController.initialize().then(() => {
  console.log("OpenID setup");
});

const app = express();

const renderStatic = (data: { title: string; json: any }) => {
  return `
  <html>
    <head>
      <title>Fediverse Auth Demo</title>
      <meta name="viewport" content="width=device-width, initial-scale=1.0">
    </head>
    <body>
      <h1>${data.title}</h1>
      <code><pre>${stripHtml(JSON.stringify(data.json, null, 2)).result}</pre></code>

      <a href="/">Home</a>
    </body>
  </html>
  `;
};

app.get("/", async (req, res) => {
  res.contentType("html").send(`
    <html>
      <head>
        <title>Fediverse Auth Demo</title>
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
      </head>
      <body>
        <h1>Fediverse Auth Demo</h1>
        <b>Query Parameters:</b>
        <code><pre>${stripHtml(JSON.stringify(req.query, null, 2)).result}</pre></code>

        <a href="/login">Test Out Fediverse Auth Login</a>
      </body>
    </html>
  `);
});

app.get("/login", async (req, res) => {
  const client = OpenIDController.get();

  res.redirect(client.getAuthorizationURL());
});

app.get("/api/callback", async (req, res) => {
  const client = OpenIDController.get();
  let exchange: OpenID.ExchangeToken;

  try {
    exchange = await client.exchangeToken(req.originalUrl);
  } catch (e) {
    if (e instanceof ResponseBodyError) {
      switch (e.error) {
        case "invalid_client":
          // this happens when we're configured with invalid credentials
          console.error(
            "OpenID is improperly configured. Cannot exchange tokens, do I have valid credentials?"
          );
          res.contentType("html").send(
            renderStatic({
              title: "Configuration Error",
              json: {
                error: "invalid_client",
              },
            })
          );
          return;
        case "invalid_grant":
          res.contentType("html").send(
            renderStatic({
              title: "invalid_grant",
              json: e.cause,
            })
          );
          return;
      }
    }

    console.log(e);

    res.contentType("html").send(
      renderStatic({
        title: "Unknown Error",
        json: {},
      })
    );
    return;
  }

  if (!exchange?.access_token) {
    return res.contentType("html").send(
      renderStatic({
        title: "Failed token exchange",
        json: {},
      })
    );
  }

  const whoami = await client.userInfo<{
    instance: {
      software: {
        name: string;
        version: string;
        logo_uri?: string;
        repository?: string;
        homepage?: string;
      };
      instance: {
        logo_uri?: string;
        banner_uri?: string;
        name?: string;
      };
    };
  }>(exchange.access_token, exchange.claims()!.sub);

  console.log(
    new Date().toISOString() + "\t" + whoami.sub + " authenticated successfully"
  );

  res.contentType("html").send(
    renderStatic({
      title: "Success",
      json: whoami,
    })
  );
});

app.listen(process.env.PORT, () => {
  console.log("Listening on :" + process.env.PORT);
});
